<?php

declare(strict_types=1);

namespace Erlage\Photogram\Data\Models\User;

use Erlage\Photogram\System;
use Erlage\Photogram\Settings;
use Erlage\Photogram\Constants\ServerConstants;
use Erlage\Photogram\Data\Tables\User\UserTable;
use Erlage\Photogram\Data\Dtos\Common\UserIdsDTO;
use Erlage\Photogram\Data\Dtos\Common\DisplayItemDTO;
use Erlage\Photogram\Data\Models\Setting\SettingEnum;
use Erlage\Photogram\Data\Models\InterfaceModelBuilder;
use Erlage\Photogram\Data\Dtos\User\UserDisplayImageDTO;
use Erlage\Photogram\Data\Dtos\User\UserMetaPushSettingsDTO;

final class UserBuilder implements InterfaceUserSetter, InterfaceModelBuilder
{
    /**
     * setter interface
     */
    use TraitUserSetters;

    /*
    |--------------------------------------------------------------------------
    | default values
    |--------------------------------------------------------------------------
    */

    /**
     * @var array
     */
    public $defaultValues;

    public function __construct()
    {
        $this -> defaultValues = self::defaultValues();
    }

    public static function defaultValues(): array
    {
        return array(
            UserTable::EMAIL         => '',
            UserTable::USERNAME      => '',
            UserTable::PASSWORD      => '',

            UserTable::DISPLAY_NAME  => '',
            UserTable::DISPLAY_BIO   => '',
            UserTable::DISPLAY_WEB   => '',

            UserTable::DISPLAY_IMAGE => (new UserDisplayImageDTO())
                -> setHost(SettingEnum::STORAGE_DISK_LOCAL)
                -> setFilespace(DisplayItemDTO::FILESPACE_PLACEHOLDER)
                -> setIdentifier(Settings::getString(ServerConstants::SS_TEXT_FILE_NAME_PLACEHOLDER_USER)),

            UserTable::META_ACCESS            => '',

            UserTable::META_IS_PRIVATE        => Settings::getString(
                ServerConstants::SS_ENUM_USER_DEFAULT_ACCOUNT_PRIVACY
            ),

            UserTable::META_IS_EMAIL_VERIFIED => UserEnum::META_IS_EMAIL_VERIFIED_NO,
            UserTable::META_LAST_ACTIVE       => System::isoDateTime(),
            UserTable::META_PUSH_SETTINGS     => UserMetaPushSettingsDTO::fromDefaultSettings(),

            UserTable::CACHE_POSTS_COUNT      => '0',
            UserTable::CACHE_FOLLOWERS_COUNT  => '0',
            UserTable::CACHE_FOLLOWINGS_COUNT => '0',
            UserTable::CACHE_BLOCKED_USER_IDS => new UserIdsDTO(),

            UserTable::STAMP_REGISTRATION     => System::isoDateTime(),
            UserTable::STAMP_LAST_UPDATE      => System::isoDateTime(),
        );
    }

    public function dispense(): UserModel
    {
        return UserModel::createFromData(
            array(
                // email

                UserTable::EMAIL => $this -> email

                ?? $this -> defaultValues[UserTable::EMAIL],

                // username

                UserTable::USERNAME => $this -> username

                ?? $this -> defaultValues[UserTable::USERNAME],

                // password

                UserTable::PASSWORD => $this -> password

                ?? $this -> defaultValues[UserTable::PASSWORD],

                // display name

                UserTable::DISPLAY_NAME => $this -> displayName

                ?? $this -> defaultValues[UserTable::DISPLAY_NAME],

                // display bio

                UserTable::DISPLAY_BIO => $this -> displayBio

                ?? $this -> defaultValues[UserTable::DISPLAY_BIO],

                // display web

                UserTable::DISPLAY_WEB => $this -> displayWeb

                ?? $this -> defaultValues[UserTable::DISPLAY_WEB],

                // display image

                UserTable::DISPLAY_IMAGE => $this -> displayImage

                ?? $this -> defaultValues[UserTable::DISPLAY_IMAGE],

                // meta access

                UserTable::META_ACCESS => $this -> metaAccess

                ?? $this -> defaultValues[UserTable::META_ACCESS],

                // meta is private

                UserTable::META_IS_PRIVATE => $this -> metaIsPrivate

                ?? $this -> defaultValues[UserTable::META_IS_PRIVATE],

                // meta is email verified

                UserTable::META_IS_EMAIL_VERIFIED => $this -> metaIsEmailVerified

                ?? $this -> defaultValues[UserTable::META_IS_EMAIL_VERIFIED],

                // meta last active

                UserTable::META_LAST_ACTIVE => $this -> metaLastActive

                ?? $this -> defaultValues[UserTable::META_LAST_ACTIVE],

                // meta push settings

                UserTable::META_PUSH_SETTINGS => $this -> metaPushSettings

                ?? $this -> defaultValues[UserTable::META_PUSH_SETTINGS],

                // cache posts count

                UserTable::CACHE_POSTS_COUNT => $this -> cachePostsCount

                ?? $this -> defaultValues[UserTable::CACHE_POSTS_COUNT],

                // cache followers count

                UserTable::CACHE_FOLLOWERS_COUNT => $this -> cacheFollowersCount

                ?? $this -> defaultValues[UserTable::CACHE_FOLLOWERS_COUNT],

                // cache followings count

                UserTable::CACHE_FOLLOWINGS_COUNT => $this -> cacheFollowingsCount

                ?? $this -> defaultValues[UserTable::CACHE_FOLLOWINGS_COUNT],

                // cache blocked user ids

                UserTable::CACHE_BLOCKED_USER_IDS => $this -> cacheBlockedUserIds

                ?? $this -> defaultValues[UserTable::CACHE_BLOCKED_USER_IDS],

                // stamp registration

                UserTable::STAMP_REGISTRATION => $this -> stampRegistration

                ?? $this -> defaultValues[UserTable::STAMP_REGISTRATION],

                // stamp last update

                UserTable::STAMP_LAST_UPDATE => $this -> stampLastUpdate

                ?? $this -> defaultValues[UserTable::STAMP_LAST_UPDATE],
            )
        );
    }
}
